// 节拍器核心类
class Metronome {
  constructor() {
    this.audioContext = null;
    this.isPlaying = false;
    this.bpm = 120;
    this.beatsPerMeasure = 4;
    this.currentBeat = 0;
    this.accentEnabled = true;
    this.nextNoteTime = 0;
    this.timerID = null;
    
    this.initElements();
    this.initAudio();
    this.bindEvents();
    this.updateBeatLights();
  }

  initElements() {
    this.bpmValue = document.getElementById('bpm-value');
    this.bpmSlider = document.getElementById('bpm-slider');
    this.bpmDecrease = document.getElementById('bpm-decrease');
    this.bpmIncrease = document.getElementById('bpm-increase');
    this.playBtn = document.getElementById('play-btn');
    this.playIcon = this.playBtn.querySelector('.play-icon');
    this.pauseIcon = this.playBtn.querySelector('.pause-icon');
    this.timeButtons = document.querySelectorAll('.time-btn');
    this.beatLightsContainer = document.getElementById('beat-lights');
    this.pendulumArm = document.querySelector('.pendulum-arm');
    this.accentToggle = document.getElementById('accent-toggle');
  }

  initAudio() {
    // 延迟初始化音频上下文，等待用户交互
  }

  createAudioContext() {
    if (!this.audioContext) {
      this.audioContext = new (window.AudioContext || window.webkitAudioContext)();
    }
    if (this.audioContext.state === 'suspended') {
      this.audioContext.resume();
    }
  }

  // 生成节拍声音
  playClick(isAccent = false) {
    if (!this.audioContext) return;

    const time = this.audioContext.currentTime;
    
    // 创建振荡器
    const osc = this.audioContext.createOscillator();
    const gainNode = this.audioContext.createGain();
    
    // 连接节点
    osc.connect(gainNode);
    gainNode.connect(this.audioContext.destination);
    
    // 设置频率和音色
    if (isAccent && this.accentEnabled) {
      // 重音：更高音调
      osc.frequency.value = 1000;
      osc.type = 'sine';
      gainNode.gain.value = 0.8;
    } else {
      // 普通节拍
      osc.frequency.value = 800;
      osc.type = 'sine';
      gainNode.gain.value = 0.5;
    }
    
    // 包络线
    gainNode.gain.setValueAtTime(gainNode.gain.value, time);
    gainNode.gain.exponentialRampToValueAtTime(0.001, time + 0.1);
    
    // 播放
    osc.start(time);
    osc.stop(time + 0.1);
  }

  // 计算下一次节拍时间
  nextNote() {
    const secondsPerBeat = 60.0 / this.bpm;
    this.nextNoteTime += secondsPerBeat;
  }

  // 调度节拍
  scheduler() {
    while (this.nextNoteTime < this.audioContext.currentTime + 0.1) {
      this.scheduleNote(this.currentBeat, this.nextNoteTime);
      this.nextNote();
      this.currentBeat = (this.currentBeat + 1) % this.beatsPerMeasure;
    }
  }

  // 调度单个音符
  scheduleNote(beatNumber, time) {
    const isAccent = beatNumber === 0;
    
    // 播放声音
    this.playClick(isAccent);
    
    // 更新视觉指示器
    this.updateVisuals(beatNumber, isAccent, time);
  }

  // 更新视觉效果
  updateVisuals(beatNumber, isAccent, time) {
    // 计算延迟
    const delay = (time - this.audioContext.currentTime) * 1000;
    
    setTimeout(() => {
      if (!this.isPlaying) return;
      
      // 更新节拍灯
      this.highlightBeat(beatNumber, isAccent);
      
      // 更新摆钟动画
      this.animatePendulum();
    }, Math.max(0, delay));
  }

  // 高亮当前节拍
  highlightBeat(beatNumber, isAccent) {
    const lights = this.beatLightsContainer.querySelectorAll('.beat-light');
    
    lights.forEach((light, index) => {
      light.classList.remove('active', 'accent');
      
      if (index === beatNumber) {
        light.classList.add('active');
        if (isAccent && this.accentEnabled) {
          light.classList.add('accent');
        }
      }
    });
  }

  // 摆钟动画
  animatePendulum() {
    // 计算摆动周期
    const swingDuration = 60 / this.bpm / 2;
    this.pendulumArm.style.setProperty('--swing-duration', `${swingDuration}s`);
    
    // 切换摆动方向
    if (this.currentBeat % 2 === 0) {
      this.pendulumArm.classList.remove('swing-right');
      this.pendulumArm.classList.add('swing-left');
    } else {
      this.pendulumArm.classList.remove('swing-left');
      this.pendulumArm.classList.add('swing-right');
    }
  }

  // 开始播放
  start() {
    if (this.isPlaying) return;
    
    this.createAudioContext();
    this.isPlaying = true;
    this.currentBeat = 0;
    this.nextNoteTime = this.audioContext.currentTime;
    
    this.updatePlayButton();
    this.scheduler();
    this.timerID = setInterval(() => this.scheduler(), 25);
  }

  // 停止播放
  stop() {
    this.isPlaying = false;
    
    if (this.timerID) {
      clearInterval(this.timerID);
      this.timerID = null;
    }
    
    this.updatePlayButton();
    this.clearVisuals();
  }

  // 切换播放状态
  toggle() {
    if (this.isPlaying) {
      this.stop();
    } else {
      this.start();
    }
  }

  // 更新播放按钮
  updatePlayButton() {
    if (this.isPlaying) {
      this.playBtn.classList.add('playing');
      this.playIcon.style.display = 'none';
      this.pauseIcon.style.display = 'inline';
    } else {
      this.playBtn.classList.remove('playing');
      this.playIcon.style.display = 'inline';
      this.pauseIcon.style.display = 'none';
    }
  }

  // 清除视觉效果
  clearVisuals() {
    const lights = this.beatLightsContainer.querySelectorAll('.beat-light');
    lights.forEach(light => {
      light.classList.remove('active', 'accent');
    });
    
    this.pendulumArm.classList.remove('swing-left', 'swing-right');
    this.pendulumArm.style.transform = 'rotate(0deg)';
  }

  // 设置 BPM
  setBPM(value) {
    this.bpm = Math.max(40, Math.min(240, value));
    this.bpmValue.textContent = this.bpm;
    this.bpmSlider.value = this.bpm;
  }

  // 设置节拍类型
  setTimeSignature(beats) {
    this.beatsPerMeasure = beats;
    this.currentBeat = 0;
    this.updateBeatLights();
    this.clearVisuals();
  }

  // 更新节拍灯数量
  updateBeatLights() {
    this.beatLightsContainer.innerHTML = '';
    
    for (let i = 0; i < this.beatsPerMeasure; i++) {
      const light = document.createElement('div');
      light.className = 'beat-light';
      light.textContent = i + 1;
      this.beatLightsContainer.appendChild(light);
    }
  }

  // 绑定事件
  bindEvents() {
    // 播放按钮
    this.playBtn.addEventListener('click', () => this.toggle());

    // BPM 滑块
    this.bpmSlider.addEventListener('input', (e) => {
      this.setBPM(parseInt(e.target.value));
    });

    // BPM 增减按钮
    this.bpmDecrease.addEventListener('click', () => {
      this.setBPM(this.bpm - 1);
    });

    this.bpmIncrease.addEventListener('click', () => {
      this.setBPM(this.bpm + 1);
    });

    // 节拍类型按钮
    this.timeButtons.forEach(btn => {
      btn.addEventListener('click', () => {
        this.timeButtons.forEach(b => b.classList.remove('active'));
        btn.classList.add('active');
        this.setTimeSignature(parseInt(btn.dataset.beats));
      });
    });

    // 重音开关
    this.accentToggle.addEventListener('change', (e) => {
      this.accentEnabled = e.target.checked;
    });

    // 键盘快捷键
    document.addEventListener('keydown', (e) => {
      if (e.code === 'Space') {
        e.preventDefault();
        this.toggle();
      } else if (e.code === 'ArrowUp') {
        e.preventDefault();
        this.setBPM(this.bpm + 1);
      } else if (e.code === 'ArrowDown') {
        e.preventDefault();
        this.setBPM(this.bpm - 1);
      }
    });
  }
}

// 初始化
document.addEventListener('DOMContentLoaded', () => {
  new Metronome();
});